
function [data_tag, data_units] = get_data_tag(out_collection_ID, N_out_fields, ...
                                               is_tavg);

% reichle - some time way back when
% reichle -  5 Feb 2009 - filled in missing units
% GDL - 7 July 2010 - renamed "tavg" to "data" 
%                     (b/c function is for "tavg" and "inst" output files)
% reichle - 23 Dec 2011: updated and revised for MERRA-Land file specs
% reichle -  9 Apr 2013: revised treatment of output collections, added SMAP L4_SM gph
% reichle - 12 Dec 2013: new out_collection_ID=7,8 (requires optional input "is_tavg")
% reichle -  6 Feb 2014: split   out_collection_ID=7 into 7 and 8 (SMAP Nature Run v03)
%                        renamed out_collection_ID=8 to   9
% reichle -  4 Mar 2014: added tpsn(1) to inst output for out_collection_ID=8
% reichle - 26 Mar 2014: added out_collection_ID=10
% reichle - 26 May 2014: out_collection_ID=6 (SMAP gph): replaced sm pctl units with volumetric units
% reichle - 27 Aug 2014: removed "out_wetness" from LDASsa nml inputs and hard-wired to "false"
%
% -----------------------------------------------------------------------------------
%
% *** For DETAILS, refer to LDASsa subroutine output_calcs(). ***********************
%
% NOTE: "N_out_fields" is NOT needed upon input *except* for output 
%        collections "4", "5", and "6".
%       For all other output collections, diagnose "N_out_fields=length(data_tag)"
%         outside of this function after it has provided "data_tag".
%
%                                         inst|tavg
% out_collection_ID =  1 :  N_out_fields =  44|44 :  legacy LDASsa output
% out_collection_ID =  2 :  N_out_fields =   6| 6 :  SMAP Nature v02
% out_collection_ID =  3 :  N_out_fields =   8| 8 :  mwRTM calibration (before Dec 2013), SMOS DA
% out_collection_ID =  4*:  N_out_fields =  50|50 :  MERRA-Land
% out_collection_ID =  5*:  N_out_fields =  59|59 :  MERRA-Land (with additional files)
% out_collection_ID =  6 :  N_out_fields =  40|40 :  SMAP L4_SM gph collection (EXCL. sm in pctl units)
%                           N_out_fields =  42|42 :  SMAP L4_SM gph collection (INCL. sm in pctl units)
% out_collection_ID =  7*:  N_out_fields =   4| 4 :  SMAP Nature v03 (2001-2009)
% out_collection_ID =  8*:  N_out_fields =   5| 6 :  SMAP Nature v03 (2010-201?)
% out_collection_ID =  9*:  N_out_fields =   6| 2 :  mwRTM calibration (Dec 2013)
% out_collection_ID = 10 :  N_out_fields =  46|46 :  legacy LDASsa output + t2m, q2m 
% __________________________________________________________________________________
% *FOOTNOTES: 
% out_collection_ID=4,5 :   tags prior to reichle-LDASsa_m2-10_p1 have one fewer output field
% out_collection_ID=7,8,9 : write *different* variables for "tavg" and "inst" output!
%
% -----------------------------------------------------------------------------------

% Up to and including LDASsa tag "reichle-LDASsa_m2-14_2" and for output 
% collections 1, 2, 3, and 10, the LDASsa nml input variable "out_wetness" 
% controlled whether soil moisture was output in "wetness" (degree of saturation) 
% or volumetric units.  
% "out_wetness" is now hard-wired to false.

out_wetness = 0;

switch out_collection_ID
  
 case {1,10}
  
  % legacy LDASsa output
  
  data_tag{ 1} = 'Tair';                data_units{ 1} = '[K]';          
  data_tag{ 2} = 'Qair';                data_units{ 2} = '[kg/kg]';          
  data_tag{ 3} = 'Psurf';               data_units{ 3} = '[Pa]';         
  data_tag{ 4} = 'RainfC';              data_units{ 4} = '[mm/d]';        
  data_tag{ 5} = 'Rainf';               data_units{ 5} = '[mm/d]';         
  data_tag{ 6} = 'Snowf';               data_units{ 6} = '[mm/d]';         
  data_tag{ 7} = 'LWdown';              data_units{ 7} = '[W/m^2]';        
  data_tag{ 8} = 'SWdown';              data_units{ 8} = '[W/m^2]';        
  data_tag{ 9} = 'Wind';                data_units{ 9} = '[m/s]';          
  
  data_tag{10} = 'capac';               data_units{10} = '[mm]';         
  data_tag{11} = 'srfexc';              data_units{11} = '[mm]';        
  data_tag{12} = 'rzexc';               data_units{12} = '[mm]';         
  data_tag{13} = 'catdef';              data_units{13} = '[mm]';        
  data_tag{14} = 'sumwesn';	        data_units{14} = '[mm]';	
  data_tag{15} = 'sumsndz';	        data_units{15} = '[m]';	
  
  data_tag{16} = 'ar1';                 data_units{16} = '[-]';           
  data_tag{17} = 'ar2';                 data_units{17} = '[-]';           
  data_tag{18} = 'asnow';               data_units{18} = '[-]';         

  if out_wetness
    
    data_tag{19} = 'sfwet';             data_units{19} = '[-]';          
    data_tag{20} = 'rzwet';             data_units{20} = '[-]';          
    data_tag{21} = 'prwet';             data_units{21} = '[-]';          

  else

    data_tag{19} = 'sfmc';              data_units{19} = '[m^3/m^3]';          
    data_tag{20} = 'rzmc';              data_units{20} = '[m^3/m^3]';          
    data_tag{21} = 'prmc';              data_units{21} = '[m^3/m^3]';          

  end

  data_tag{22} = 'tsurf';               data_units{22} = '[K]';         
  data_tag{23} = 'tp1';                 data_units{23} = '[K]';           
  data_tag{24} = 'tpN';		        data_units{24} = '[K]';		
  data_tag{25} = 'tpsn1';		data_units{25} = '[K]';
  data_tag{26} = 'tpsnN';		data_units{26} = '[K]';
  data_tag{27} = 'shflux';              data_units{27} = '[W/m^2]';        
  data_tag{28} = 'lhflux';              data_units{28} = '[W/m^2]';        
  data_tag{29} = 'ghflux';              data_units{29} = '[W/m^2]';        
  data_tag{30} = 'evap';                data_units{30} = '[mm/d]';          
  data_tag{31} = 'eint';                data_units{31} = '[mm/d]';          
  data_tag{32} = 'eveg';                data_units{32} = '[mm/d]';          
  data_tag{33} = 'esoi';                data_units{33} = '[mm/d]';          
  data_tag{34} = 'esno';                data_units{34} = '[mm/d]';          
  data_tag{35} = 'runoff';              data_units{35} = '[mm/d]';        
  data_tag{36} = 'runsrf';              data_units{36} = '[mm/d]';        
  data_tag{37} = 'bflow';               data_units{37} = '[mm/d]';         
  data_tag{38} = 'snmelt';              data_units{38} = '[mm/d]';        
  data_tag{39} = 'lwup';                data_units{39} = '[W/m^2]';          
  data_tag{40} = 'swup';                data_units{40} = '[W/m^2]';          
  data_tag{41} = 'qinfil';              data_units{41} = '[kg/m^2/s]';        
  data_tag{42} = 'totalb';              data_units{42} = '[-]';        
  data_tag{43} = 'waterbal';	        data_units{43} = '[mm/d]';	
  data_tag{44} = 'energybal';           data_units{44} = '[W/m^2]';

  if out_collection_ID==10
    
    data_tag{45} = 't2m';               data_units{45} = '[K]';          
    data_tag{46} = 'q2m';               data_units{46} = '[kg/kg]';         
    
  end

  
  % ----------------------------------------------------------------
  
 case {2,3}
  
  % case 2: SMAP Nature Run v02 (~2011)
  % case 3: mwRTM calibration (Gabrielle De Lannoy, before Dec 2013), SMOS DA
  
  % for more information see file DOCUMENTATION/SMAP_Nature_Run_variables.xls
  
  if out_wetness
    
    data_tag{ 1} = 'sfwet';             data_units{1} = '[-]';       
    data_tag{ 2} = 'rzwet';             data_units{2} = '[-]';       
    
  else
    
    data_tag{ 1} = 'sfmc';              data_units{1} = '[m^3/m^3]';   
    data_tag{ 2} = 'rzmc';              data_units{2} = '[m^3/m^3]';     
    
  end
  
  data_tag{ 3} = 'tsurf';               data_units{ 3} = '[K]';         
  data_tag{ 4} = 'tp1';                 data_units{ 4} = '[K]';           
  data_tag{ 5} = 'sumwesn';             data_units{ 5} = '[mm]';         
  data_tag{ 6} = 'Prectotl';            data_units{ 6} = '[mm/d]';         

  if out_collection_ID==3
    
    data_tag{ 7} = 'Tair';              data_units{ 7} = '[K]';          
    data_tag{ 8} = 'capac';             data_units{ 8} = '[mm]';         
    
  end
  
  % ----------------------------------------------------------------
    
 case {4,5}

  % CVS tags: reichle-LDASsa_m2-10, GEOSldas-MERRALand
  %  
  % case 4, N_out_fields=49: MERRA-Land file specs 
  % case 5, N_out_fields=58: MERRA-Land file specs plus select additional outputs

  % CVS tags: reichle-LDASsa_m2-10_p1 and thereafter
  %
  % case 4, N_out_fields=50: MERRA-Land file specs 
  % case 5, N_out_fields=59: MERRA-Land file specs plus select additional outputs

  if     N_out_fields==50 | N_out_fields==59
  
    ioff = 1;

  elseif N_out_fields==49 | N_out_fields==58

    ioff = 0;

  else

    error(['get_data_tag.m: unknown N_out_fields=',num2str(N_out_fields)])

  end

  data_tag{ 1     } = 'GRN';                 data_units{ 1     } = '[Fraction]';          
  data_tag{ 2     } = 'LAI';                 data_units{ 2     } = '[m^2/m^2]';          
                  		                           			   
  data_tag{ 3     } = 'GWETPROF';            data_units{ 3     } = '[Fraction]';         
  data_tag{ 4     } = 'GWETROOT';            data_units{ 4     } = '[Fraction]';        
  data_tag{ 5     } = 'GWETTOP';             data_units{ 5     } = '[Fraction]';         
                  		
  data_tag{ 6     } = 'PRMC';                data_units{ 6     } = '[m^3/m^3]';         
  data_tag{ 7     } = 'RZMC';                data_units{ 7     } = '[m^3/m^3]';        
  data_tag{ 8     } = 'SFMC';                data_units{ 8     } = '[m^3/m^3]';

  if N_out_fields==50 | N_out_fields==59
  
    data_tag{ 9     }='TSURF';               data_units{ 9     } = '[K]';          

  end

  data_tag{ 9+ioff} = 'TPSNOW';              data_units{ 9+ioff} = '[K]';          
  data_tag{10+ioff} = 'TUNST';               data_units{10+ioff} = '[K]';         
  data_tag{11+ioff} = 'TSAT';                data_units{11+ioff} = '[K]';        
  data_tag{12+ioff} = 'TWLT';                data_units{12+ioff} = '[K]';   

  data_tag{13+ioff} = 'TSOIL1';              data_units{13+ioff} = '[K]';        
  data_tag{14+ioff} = 'TSOIL2';	             data_units{14+ioff} = '[K]';	
  data_tag{15+ioff} = 'TSOIL3';	             data_units{15+ioff} = '[K]';	
  data_tag{16+ioff} = 'TSOIL4';              data_units{16+ioff} = '[K]';           
  data_tag{17+ioff} = 'TSOIL5';              data_units{17+ioff} = '[K]';           
  data_tag{18+ioff} = 'TSOIL6';              data_units{18+ioff} = '[K]';

  data_tag{19+ioff} = 'PRECSN';              data_units{19+ioff} = '[kg/m^2/s]';          
  data_tag{20+ioff} = 'PRECTO';              data_units{20+ioff} = '[kg/m^2/s]';          
  
  data_tag{21+ioff} = 'SNOMAS';              data_units{21+ioff} = '[kg/m^2]'; 
  data_tag{22+ioff} = 'SNODP';               data_units{22+ioff} = '[m]';
  
  data_tag{23+ioff} = 'EVPSOIL';             data_units{23+ioff} = '[W/m^2]';           
  data_tag{24+ioff} = 'EVPTRNS';             data_units{24+ioff} = '[W/m^2]';		
  data_tag{25+ioff} = 'EVPINTR';	     data_units{25+ioff} = '[W/m^2]';
  data_tag{26+ioff} = 'EVPSBLN';	     data_units{26+ioff} = '[W/m^2]';
  
  data_tag{27+ioff} = 'RUNOFF';              data_units{27+ioff} = '[kg/m^2/s]';        
  data_tag{28+ioff} = 'BASEFLOW';            data_units{28+ioff} = '[kg/m^2/s]';        
  data_tag{29+ioff} = 'SMLAND';              data_units{29+ioff} = '[kg/m^2/s]';        
  data_tag{30+ioff} = 'QINFIL';              data_units{30+ioff} = '[kg/m^2/s]';          
  
  data_tag{31+ioff} = 'FRUNST';              data_units{31+ioff} = '[Fraction]';          
  data_tag{32+ioff} = 'FRSAT';               data_units{32+ioff} = '[Fraction]';          
  data_tag{33+ioff} = 'FRSNO';               data_units{33+ioff} = '[Fraction]';          
  data_tag{34+ioff} = 'FRWLT';               data_units{34+ioff} = '[fraction]';          
  
  data_tag{35+ioff} = 'PARDF';               data_units{35+ioff} = '[W/m^2]';        
  data_tag{36+ioff} = 'PARDR';               data_units{36+ioff} = '[W/m^2]';        
  
  data_tag{37+ioff} = 'SHLAND';              data_units{37+ioff} = '[W/m^2]';         
  data_tag{38+ioff} = 'LHLAND';              data_units{38+ioff} = '[W/m^2]';        
  data_tag{39+ioff} = 'EVLAND';              data_units{39+ioff} = '[kg/m^2/s]';   
  
  data_tag{40+ioff} = 'LWLAND';              data_units{40+ioff} = '[W/m^2]';          
  data_tag{41+ioff} = 'SWLAND';              data_units{41+ioff} = '[W/m^2]';        
  
  data_tag{42+ioff} = 'GHLAND';              data_units{42+ioff} = '[W/m^2]';        
  
  data_tag{43+ioff} = 'TWLAND';	             data_units{43+ioff} = '[kg/m^2]';	
  data_tag{44+ioff} = 'TELAND';              data_units{44+ioff} = '[J/m^2]';
  data_tag{45+ioff} = 'WCHANGE';             data_units{45+ioff} = '[kg/m^2/s]';
  data_tag{46+ioff} = 'ECHANGE';             data_units{46+ioff} = '[W/m^2]';
  
  data_tag{47+ioff} = 'SPLAND';              data_units{47+ioff} = '[W/m^2]';
  data_tag{48+ioff} = 'SPWATR';              data_units{48+ioff} = '[kg/m^2/s]';
  data_tag{49+ioff} = 'SPSNOW';              data_units{49+ioff} = '[W/m^2]';
  
  if N_out_fields==58 | N_out_fields==59
    
    data_tag{50+ioff}='TLML';                data_units{50+ioff} = '[K]';
    data_tag{51+ioff}='QLML';                data_units{51+ioff} = '[kg/kg]';
    data_tag{52+ioff}='LWGAB';               data_units{52+ioff} = '[W/m^2]';
    data_tag{53+ioff}='SWGDN';               data_units{53+ioff} = '[W/m^2]';
    
    data_tag{54+ioff}='SRFEXC';              data_units{54+ioff} = '[kg/m^2]';
    data_tag{55+ioff}='RZEXC';               data_units{55+ioff} = '[kg/m^2]';
    data_tag{56+ioff}='CATDEF';              data_units{56+ioff} = '[kg/m^2]';
    
    data_tag{57+ioff}='WINCR';               data_units{57+ioff} = '[kg/m^2]';
    data_tag{58+ioff}='EINCR';               data_units{58+ioff} = '[J/m^2]';
    
  end

 case {6}
  
  % SMAP L4_SM gph collection

  % for more information see SMAP L4_SM Data Product Specification Document (PSD; revised Jun 2014) 
  %  
  % NOTE: rootzone and profile soil moisture outputs in units of percentiles are appended in post-processing
  %
  % reichle, 26 May 2014 - revised output specs:
  %                         - replaced sm in pctl units (fields #1-3) with sm in volumetric units
  %                         - sm in pctl units to be appended as fields #41-42 in post-processing

  data_tag{ 1} = 'sm_surface';                         data_units{ 1} = '[m3 m-3]';
  data_tag{ 2} = 'sm_rootzone';                        data_units{ 2} = '[m3 m-3]';
  data_tag{ 3} = 'sm_profile';                         data_units{ 3} = '[m3 m-3]';
  data_tag{ 4} = 'sm_surface_wetness';                 data_units{ 4} = '[dimensionless]';
  data_tag{ 5} = 'sm_rootzone_wetness';                data_units{ 5} = '[dimensionless]';
  data_tag{ 6} = 'sm_profile_wetness';                 data_units{ 6} = '[dimensionless]';
  data_tag{ 7} = 'surface_temp';                       data_units{ 7} = '[K]';          
  data_tag{ 8} = 'soil_temp_layer1';                   data_units{ 8} = '[K]';           
  data_tag{ 9} = 'soil_temp_layer2';                   data_units{ 9} = '[K]';           
  data_tag{10} = 'soil_temp_layer3';                   data_units{10} = '[K]';           
  data_tag{11} = 'soil_temp_layer4';                   data_units{11} = '[K]';           
  data_tag{12} = 'soil_temp_layer5';                   data_units{12} = '[K]';           
  data_tag{13} = 'soil_temp_layer6';                   data_units{13} = '[K]';           
  data_tag{14} = 'snow_mass';                          data_units{14} = '[kg m-2]';          
  data_tag{15} = 'snow_depth';                         data_units{15} = '[m]';           
  data_tag{16} = 'land_evapotranspiration_flux';       data_units{16} = '[kg m-2 s-1]';          
  data_tag{17} = 'overland_runoff_flux';               data_units{17} = '[kg m-2 s-1]';          
  data_tag{18} = 'baseflow_flux';                      data_units{18} = '[kg m-2 s-1]';          
  data_tag{19} = 'snow_melt_flux';                     data_units{19} = '[kg m-2 s-1]';          
  data_tag{20} = 'soil_water_infiltration_flux';       data_units{20} = '[kg m-2 s-1]';          
  data_tag{21} = 'land_fraction_saturated';            data_units{21} = '[dimensionless]';          
  data_tag{22} = 'land_fraction_unsaturated';          data_units{22} = '[dimensionless]';          
  data_tag{23} = 'land_fraction_wilting';              data_units{23} = '[dimensionless]';          
  data_tag{24} = 'land_fraction_snow_covered';         data_units{24} = '[dimensionless]';          
  data_tag{25} = 'heat_flux_sensible';                 data_units{25} = '[W m-2]';          
  data_tag{26} = 'heat_flux_latent';                   data_units{26} = '[W m-2]';          
  data_tag{27} = 'heat_flux_ground';                   data_units{27} = '[W m-2]';          
  data_tag{28} = 'net_downward_shortwave_flux';        data_units{28} = '[W m-2]';          
  data_tag{29} = 'net_downward_longwave_flux';         data_units{29} = '[W m-2]';          
  data_tag{30} = 'radiation_shortwave_downward_flux';  data_units{30} = '[W m-2]';          
  data_tag{31} = 'radiation_longwave_absorbed_flux';   data_units{31} = '[W m-2]';          
  data_tag{32} = 'precipitation_total_surface_flux';   data_units{32} = '[kg m-2 s-1]';          
  data_tag{33} = 'snowfall_surface_flux';              data_units{33} = '[kg m-2 s-1]';          
  data_tag{34} = 'surface_pressure';                   data_units{34} = '[Pa]';          
  data_tag{35} = 'height_lowatmmodlay';                data_units{35} = '[m]';          
  data_tag{36} = 'temp_lowatmmodlay';                  data_units{36} = '[K]';          
  data_tag{37} = 'specific_humidity_lowatmmodlay';     data_units{37} = '[kg kg-1]';          
  data_tag{38} = 'windspeed_lowatmmodlay';             data_units{38} = '[m s-1]';          
  data_tag{39} = 'vegetation_greenness_fraction';      data_units{39} = '[dimensionless]';          
  data_tag{40} = 'leaf_area_index';                    data_units{40} = '[m2 m-2]';          

  % rootzone and profile sm in pctl units are appended in post-processing!
  % - reichle, 26 May 2014

  if N_out_fields==42

    data_tag{41} = 'sm_rootzone_pctl';                 data_units{41} = '[percent]';
    data_tag{42} = 'sm_profile_pctl';                  data_units{42} = '[percent]';
  
  end

  % ----------------------------------------------------------------
  
 case {7,8}
  
  % case 7:  SMAP Nature v03 (2001-2009)
  % case 8:  SMAP Nature v03 (2010-201?)    

  if out_wetness
    
    error('out_collection_ID=7,8 (SMAP Nature v03) inconsistent with out_wetness=.true.')
    
  end

  if ~exist('is_tavg','var')
    
    error('out_collection_ID=7,8 (SMAP Nature v03) requires input of "is_tavg"')
    
  end

  if ~is_tavg

    data_tag{  1} = 'surface_temp';                      data_units{ 1} = '[K]';          
    data_tag{  2} = 'soil_temp_layer1';                  data_units{ 2} = '[K]';           
    data_tag{  3} = 'tb_h';                              data_units{ 3} = '[K]';  % 40 deg, 1.41 GHz       
    data_tag{  4} = 'tb_v';                              data_units{ 4} = '[K]';  % 40 deg, 1.41 GHz         

    if out_collection_ID==8

      data_tag{5} = 'snow_temp_layer1';                  data_units{ 5} = '[K]';          

    end


  else

    data_tag{  1} = 'sm_surface';                        data_units{ 1} = '[m3 m-3]';
    data_tag{  2} = 'sm_rootzone';                       data_units{ 2} = '[m3 m-3]';
    data_tag{  3} = 'sm_profile';                        data_units{ 3} = '[m3 m-3]';
    data_tag{  4} = 'soil_temp_layer1';                  data_units{ 4} = '[K]';           

    if out_collection_ID==8

      data_tag{5} = 'snow_mass';                         data_units{ 5} = '[kg m-2]';          
      data_tag{6} = 'precipitation_total_surface_flux';  data_units{ 6} = '[kg m-2 s-1]';   

    end

  end
  
  % ----------------------------------------------------------------

 case {9}
  
  % case 9: for L-band mwRTM calibration (Dec 2013)

    
  if out_wetness
    
    error('out_collection_ID=9 inconsistent with out_wetness=.true.')
    
  end

  if ~exist('is_tavg','var')
    
    error('out_collection_ID=9 requires input of "is_tavg"')
    
  end

  if ~is_tavg

    data_tag{ 1} = 'sm_surface';                         data_units{ 1} = '[m3 m-3]';
    data_tag{ 2} = 'sm_rootzone';                        data_units{ 2} = '[m3 m-3]';
    data_tag{ 3} = 'surface_temp';                       data_units{ 3} = '[K]';          
    data_tag{ 4} = 'soil_temp_layer1';                   data_units{ 4} = '[K]';           
    data_tag{ 5} = 'temp_lowatmmodlay';                  data_units{ 5} = '[K]';
    data_tag{ 6} = 'leaf_area_index';                    data_units{ 6} = '[m2 m-2]';

  else

    data_tag{ 1} = 'snow_mass';                          data_units{ 1} = '[kg m-2]';          
    data_tag{ 2} = 'precipitation_total_surface_flux';   data_units{ 2} = '[kg m-2 s-1]';   

  end
  
  % ----------------------------------------------------------------

 otherwise
  
  error(['get_data_tag.m: unknown out_collection_ID=',num2str(out_collection_ID)])
  
end

% ============= EOF =================================================
